/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true, node: true */

(function () {
    "use strict";
    
    var _  = require("underscore");

    var DocinfoUtils = function () {};

    DocinfoUtils.prototype.LAYER = "layer";
    DocinfoUtils.prototype.DOCUMENT = "document";

    /**
     * Performs a depth-first traversal of docinfo.
     *
     * @param {Object} root
     * @param {Function} visitorFunction
     * @param {Object} [context]
     */
    DocinfoUtils.prototype.visitLayers = function (root, visitorFunction, context) {
        if (root && root.layers) {
            root.layers.forEach(function (layer) {
                // If the visitor function returns false, don't visit the children of the layer.
                var visitChildren = visitorFunction(layer, context);
                if (!visitChildren) {
                    return;
                }

                this.visitLayers(layer, visitorFunction, context);
            }.bind(this));
        }
    };

    /**
     * Returns each selected layer, except if the layer's siblings are all selected. In other words, if the user
     * selects a group and all of its children are also selected, we only export the group. If only some of the group's
     * children are selected, we export both the selected group and the selected children.
     *
     * @param {Object} docinfo
     * @param {Object} [parentArtboardMapToPopulate] Optional hash map to populate with layer ids to parent artboards.
     * @return {Object} result.parentArtboardMap A map of the layer ids in result.layers to parent artboards.
     */
    DocinfoUtils.prototype.getLayersForExport = function (docinfo, parentArtboardMapToPopulate) {
        var layersForExport = [],
            selectedLayerIdsMap = {};

        // Put the selected layer ids in a hash map for quick lookups.
        docinfo._selectionById.forEach(function (selectedLayerId) {
            selectedLayerIdsMap[selectedLayerId] = true;
        });

        // Visit each layer and add it to the array if it's selected.
        this.visitLayers(docinfo, function (layer, context) {
            var isLayerForExport = this._addLayerForExportIfNeeded(layer, selectedLayerIdsMap, layersForExport);
            this._addParentArtboardIfNeeded(layer, context, isLayerForExport, parentArtboardMapToPopulate);
            return this._shouldVisitChildren(layer, isLayerForExport, selectedLayerIdsMap);
        }.bind(this), {});

        return layersForExport;
    };

    DocinfoUtils.prototype._addLayerForExportIfNeeded = function (layer, selectedLayerIdsMap, layersForExport) {
        var isSelected = selectedLayerIdsMap[layer.id];
        if (isSelected) {
            layersForExport.push(layer);
        }
        return isSelected;
    };

    DocinfoUtils.prototype._addParentArtboardIfNeeded = function (layer, context, isLayerForExport, parentArtboardMapToPopulate) {
        if (!parentArtboardMapToPopulate) {
            // No map available to populate.
            return;
        }

        if (this.layerIsArtboard(layer)) {
            context.parentArtboard = layer;
        } else if (isLayerForExport) {
            parentArtboardMapToPopulate[layer.id] = context.parentArtboard;
        }
    };

    DocinfoUtils.prototype._shouldVisitChildren = function (layer, isLayerForExport, selectedLayerIdsMap) {
        // If all of the exported layer's children are also selected, do not visit them.
        return isLayerForExport ? !this._allChildrenAreSelected(layer, selectedLayerIdsMap) : true;
    };

    DocinfoUtils.prototype._allChildrenAreSelected = function (parentLayer, selectedLayerIdsMap) {
        return _(parentLayer.layers).every(function (childLayer) {
            return selectedLayerIdsMap[childLayer.id];
        });
    };

    /**
     * See if the document has artboards. If so, return an array of them; if not, return an empty array.
     *
     * @param {Object} docinfo
     * @return {Array} An array of layers.
     */
    DocinfoUtils.prototype.getArtboards = function (docinfo) {
        // Artboards will be at the top level if they exist.
        return _(docinfo.layers).filter(function (layer) {
            return DocinfoUtils.prototype.layerIsArtboard(layer);
        });
    };

    /**
     * See if the specified layer is an artboard, based on whether or not it has an artboard property.
     */
    DocinfoUtils.prototype.layerIsArtboard = function (layer) {
        return (_.has(layer, "artboard") && _.has(layer.artboard, "top"));
    };
    
    DocinfoUtils.prototype.getSourceObjectType = function (sourceObject) {
        return sourceObject.file ? this.DOCUMENT : this.LAYER;
    };

    module.exports = new DocinfoUtils();
}());
